#include "process.h"
#include "frame.h"
#include "symtab.h"
#include "symbol.h"
#include "core.h"
#include "mipssymtab.h"
#include "asm.h"
#include "bpts.h"
#include "mipsasm.h"
#include "mipscore.h"
SRCFILE("mipscore.c")

class FrameList {
	friend	MipsCore;
protected:
	FpFrame	fpf;
	FrameList *next;
		FrameList(FrameList* = 0);
		~FrameList();
};

FrameList::FrameList(FrameList *p)
{
	if (p)
		p->next = this;
}

FrameList::~FrameList()
{
	if (next)
		delete next;
}

int MipsCore::REG_SP()			{ return 29; }
int MipsCore::REG_PC()			{ return 32; }
int MipsCore::nregs()			{ return 39; }
long MipsCore::instrafterjsr()		{ return pc()+8; }
Asm *MipsCore::newAsm()			{ return new MipsAsm(this); }
long MipsCore::fp()			{ return regpeek(REG_SP()); }
MipsSymTab *MipsCore::mipsymtab()	{ return (MipsSymTab*)_symtab; }

void MipsCore::newSymTab(long reloc)
{
	 _symtab = new MipsSymTab(this, stabfd, _symtab, reloc);
}

MipsCore::MipsCore()
{
	stackdir = GROWDOWN;
	memlayout = MSBFIRST;
	bptsize = 4;
}

char *MipsCore::regname(int r)
{
	static char *regnames[] = {
		"zero", "at", "v0", "v1", "a0", "a1", "a2", "a3",
		"t0", "t1", "t2", "t3", "t4", "t5", "t6", "t7",
		"s0", "s1", "s2", "s3", "s4", "s5", "s6", "s7",
		"t8", "t9", "k0", "k1", "gp", "sp", "s8", "ra",
                "pc", "cause", "badvaddr", "mmhi", "mmlo", "fpcpcsr", "fpceir"
	};
	if (r < nregs())
		return regnames[r];
	else
		return 0;
}

long MipsCore::saved(Frame *f, int reg, int sz)
{
	int offset = 0;
	int i;
	
	if (((f->regsave >> reg) & 1) != 0) {
		for (i=31; i>=0; --i) {
			if (i == reg)
				break;
			if (((f->regsave >> i) & 1) != 0)
				offset -= 4;
		}
	} else
		return 0;
	return (f->fp + f->regbase + offset + 4 - sz);
}

Frame MipsCore::frameabove(long _fp)
{
	register int i;

	if (!_fp)
		return *callstk->fpf[0].frame;
	for (i = 0; i < callstk->size-1; i++) {
		if( _fp == callstk->fpf[i].fp )
			break;
	}
	return *callstk->fpf[i+1].frame;
}

CallStk *MipsCore::callstack()
{
	int		i, cnt;
	int		pcreg;
	long		pcaddr;
	long		pct = pc();
	Func		*func = (Func*)_symtab->loctosym(U_FUNC, pct);

	if (!func)
		return 0;
	long fp = regpeek(mipsymtab()->framereg(func)) +
		mipsymtab()->framesize(func);
	FrameList *head = new FrameList;
	FrameList *flp = head;
	for (cnt = 0; cnt < 1000; cnt++) {
		flp->fpf.fp = fp;
		flp->fpf.func = func;
		flp->fpf.frame = new Frame(this);
		flp->fpf.frame->func = func;
		flp->fpf.frame->pc = pct;
		flp->fpf.frame->fp = fp;
		flp->fpf.frame->ap = fp;
		flp->fpf.frame->regbase = mipsymtab()->regoffset(func);
		flp->fpf.frame->regsave = mipsymtab()->regmask(func);
		// If no regs saved and we are not in a leaf node
		// something is wrong
		if (cnt && !flp->fpf.frame->regsave)
			break;
		pcreg = (int)mipsymtab()->pcreg(func);
		if (pcaddr = saved(flp->fpf.frame, pcreg, 4))
			pct = peek(pcaddr)->lng;
		else
			pct = regpeek(pcreg);
		if (!pct || !(func = (Func*)_symtab->loctosym(U_FUNC, pct)))
			break;
		fp += mipsymtab()->framesize(func);
		if (cnt && fp == flp->fpf.fp)
			break;
		flp = new FrameList(flp);
	}
	if (cnt)
		callstk = new CallStk(cnt, this);
	else
		callstk = 0;
	for (flp = head, i = 0; i < cnt; i++ ) {
		callstk->fpf[i] = flp->fpf;
		callstk->fpf[i].frame->level = i;
		flp = flp->next;
	}
	delete flp->fpf.frame;
	delete head;
	return callstk;
		
}

const long MIPS_INSTR_MASK = 0xFC000000, MIPS_JAL = 0x0C000000;
const long MIPS_JALR_MASK = 0xFC1F07FF, MIPS_JALR = 0x00000009;
int MipsCore::atjsr(long pc)
{
	long instr = peek(pc)->lng;
	return ((instr & MIPS_INSTR_MASK) == MIPS_JAL ||
		(instr & MIPS_JALR_MASK) == MIPS_JALR);
}

char *MipsCore::popcallstack()
{
	char *error;
	static Trap *t;

	if(!t)
		t = new Trap(new Stmt(0,0,0),0);
	if (atsyscall())
		return "pop callstack: process in system call";
	if (!callstk || callstk->size < 2)
		return "pop callstack: not enough frames on stack";
	t->stmt->range.lo = callstk->fpf[1].frame->pc;
	if (error = laybpt(t))
		return error;
	// Don't compare fp on leaf nodes, it doesn't move
	if (!callstk->fpf[0].frame->regsave)
		error = dostep(0,0,0);
	else  {
		long fpt, fp0 = fp();
		while (!(error = dostep(0,0,0)) &&
			fpvalid(fpt = fp()) &&
			fpt <= fp0 &&
			!(error = liftbpt(t)) &&
			!(error = dostep(0,0,1)) &&
			!(error = laybpt(t)))
				;
	}
	if (error)
		liftbpt(t);
	else
		error = liftbpt(t);
	return error;
}

char *MipsCore::stepprolog()
{
	char *err;
	long cpc = pc();
	Func *f = (Func*)_symtab->loctosym(U_FUNC, cpc);
	if (!f)
		return 0;
	Stmt *s = f->stmt(cpc);
	if (s && f->range.lo == s->range.lo) {
		process()->stmtstep(1);
		return 0;
	}
	int nstep = mipsymtab()->prologinstr(f);
	nstep -= (int)(cpc - f->range.lo) >> 2;
	while(nstep-- > 0 && !(err = step()))
		;
	return err;
}
